/*
* Creation date : Thu Oct 11 15:12:11 2007
* Last modified : %modify_time%
*/
/** @file
* \brief This file contains implementation of functions that
* provides a error description retreiving.
*
* \version LLF_error.c#1:csrc:1
* \author Alexander Masiuk
* \remarks Copyright (C) 2007 by Discretix Technologies Ltd.
* All Rights reserved
*/

/************************ Include Files ***********************/

#include "LLF_error.h"

/************************ Defines *****************************/

#define ERRORMODULE_FROM_ERRORITEM_ARRAY(ei_array) { ei_array, (sizeof(ei_array) / sizeof(struct ErrorItem)) }

/*             Default error messages definition              */
#define CE2_ERROR_BASE_MESSAGE					"The global base error"
#define CE2_OK_MESSAGE							"Ok (success)"
#define CE2_ERROR_UNDEFINED_ERROR_MESSAGE		"Undefined error code"
#define CE2_ERROR_UNDEFINED_ERROR_LAYER_MESSAGE	"Undefined error code: unknown error layer"
#define CE2_ERROR_UNDEFINED_ERROR_MODULE_MESSAGE "Undefined error code: unknown error module layer"

/*             Default error messages definition              */
#define CE2_DATA_IN_PTR_ERROR_MESSAGE			"Input data pointer is invalid: "\
												"Input data must be defined for this operation. "
#define CE2_DATA_IN_SIZE_ERROR_MESSAGE			"Input data size is invalid. "
#define CE2_DATA_OUT_PTR_ERROR_MESSAGE			"Output data pointer is invalid: "\
												"Output data must be defined for this operation. "
#define CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE		"Output data length pointer is invalid: "\
												"Output data length must be defined for this operation. "
#define CE2_DATA_OUT_SIZE_ERROR_MESSAGE			"Output data length is invalid. "
#define CE2_KEY_PTR_ERROR_MESSAGE				"Key pointer is invalid: "\
												"A pointer to the user's key buffer must be defined. "
#define CE2_KEY_SIZE_PTR_ERROR_MESSAGE			"Key size pointer is invalid: "\
												"A pointer to the user's key size buffer must be defined. "
#define CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE		"Private key pointer is invalid: "\
												"A pointer to the user's private key buffer must be defined. "
#define CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE		"Public key pointer is invalid: "\
												"A pointer to the user's public key buffer must be defined. "
#define CE2_KEY_SIZE_ERROR_MESSAGE				"Key size is invalid. "
#define CE2_MODULUS_PTR_ERROR_MESSAGE			"Modulus pointer is invalid. "
#define CE2_MODULUS_SIZE_ERROR_MESSAGE			"Modulus size is invalid: "\
												"Mudulus size must be between 64 and 264 bytes and divisible by 4."
#define CE2_MODULUS_MS_BIT_ERROR_MESSAGE		"Modulus MS bit is invalid: MS bit must be equal to 1."
#define CE2_EXPONENT_PTR_ERROR_MESSAGE			"Exponent pointer is invalid. "
#define CE2_EXPONENT_SIZE_ERROR_MESSAGE			"Exponent size is invalid: "\
												"Exponent size must be less or equals to 4 bytes. "
#define CE2_PUB_EXPONENT_PTR_ERROR_MESSAGE		"Public exponent pointer is invalid. "
#define CE2_PRV_EXPONENT_PTR_ERROR_MESSAGE		"Private exponent pointer is invalid. "
#define CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE	"Encrypt/Decrypt operation is invalid in current content. "
#define CE2_HASH_MODE_ERROR_MESSAGE				"Hash mode is invalid. "
#define CE2_RSA_HASH_MODE_ERROR_MESSAGE			"Hash mode is invalid: "\
												"Supported modes are "\
												"MD5, SHA1/224/256/384/512 and modes after "\
												"MD5, SHA1/224/256/384/512, HASH_NOT_KNOWN or "\
												"NO_HASH_mode for PKCS1 v1.5 Encrypt and Decrypt."
#define CE2_OTF_HASH_CHAIN_MODE_ERROR_MESSAGE	"Hash chain mode is invalid: "\
												"Supported only PainText, ChiperFirstBlock, ChiperData and NoHASH modes."
#define CE2_PRIME_PTR_ERROR_MESSAGE				"Prime pointer is invalid. "
#define CE2_DER_FUNC_MODE_ERROR_MESSAGE			"The type of function to derive the secret key to the key data is invalid. "\
												"There are ASN.1-based function or HASH concatenation function modes."
#define CE2_DOMAIN_ID_ERROR_MESSAGE				"DomainId is invalid. "
#define CE2_MGF_MODE_ERROR_MESSAGE				"The mask generation function is invalid in current context: "\
												"Supported modes are PKCS1_MGF1 and PKCS1_NO_MGF. "\
												"PKCS#1 v2.1 defines MGF1."
#define CE2_PKCS_VERSION_ERROR_MESSAGE			"The PKCS#1 version is invalid: "\
												"Supported Ver 1.5 or 2.1, according to the functionality required."
#define CE2_KEK_PTR_ERROR_MESSAGE				"KEK pointer is invalid: "\
												"Key Encryption Key, an octet string of length 16 must be defined."
#define CE2_WRAPPED_PTR_ERROR_MESSAGE			"Wrapped key pointer is invalid: "\
												"Wrapped Key, an octet string of length 40 bytes must be defined."

/************************ Enums *******************************/
/************************ Typedefs ****************************/

/**
****************************************************************
* Structure Name: 
* ErrorItem
*
* \brief \b 
* Description:
*  This structure contains error code and error description.
***************************************************************/
struct ErrorItem
{
	/* Error code */
	CE2Error_t	error;
	/* Error description */
	DxUint8_t*	description;
}; /* End of ErrorItem */

/**
****************************************************************
* Structure Name: 
* ErrorModule
*
* \brief \b 
* Description:
*  This structure contains errors of specified module.
***************************************************************/
struct ErrorModule
{
	/* Error code */
	struct ErrorItem* errors;
	/* Error description */
	DxUint32_t errors_count;
}; /* End of ErrorModule */

/************************ Global Data *************************/

/**
 * \brief The definitions of the errors used for the different modules
 */

/**
****************************************************************
* Data Name: 
* AES_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the AES module.
***************************************************************/
struct ErrorItem AES_Errors[] =
{
/********************/
/* AES module error */
/********************/
/* AES module on the CE2 layer base address - 0x00F00000 */
{ CE2_AES_MODULE_ERROR_BASE,					"AES module on the CE2 layer base error" },

/* AES module on the LLF layer base address -  0x00F10000 */
{ CE2_LLF_AES_MODULE_ERROR_BASE,				"AES module on the LLF layer base error" },

/* DX_AES() error codes */
{ CE2_AES_DATA_IN_PTR_ERROR,					CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_AES_DATA_OUT_PTR_ERROR,					CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_AES_DATA_IN_SIZE_ERROR,					CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"The size of the input data (must be NOT 0 and must be divisible by 16 bytes, "
												"besides XCBC and CMAC)" },
{ CE2_AES_IV_COUNTER_PTR_ERROR,					"IV counter pointer is invalid:"
												"This parameter is the buffer of the IV or counters on mode CTR."
												"In ECB, XCBC, CMAC mode this parameter is not used."
												"In CBC and MAC modes this parameter should contain the IV values."
												"In CTR mode this value should contain the init counter."
												"In XCBC and CMAC modes it may be NULL" },
{ CE2_AES_KEY_PTR_ERROR,						CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_AES_KEY_SIZE_ERROR,						CE2_KEY_SIZE_ERROR_MESSAGE
												"The size of the KEY used by the AES: 128/192/256 bits keys" },
{ CE2_AES_OPERATION_MODE_ERROR,					"Operation mode is not supported" },
{ CE2_AES_ENCRYPT_DECRYPT_FLAG_ERROR,			CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE
												"In XCBC and CMAC modes it must be Encrypt." },

/* CE2_AES_Wrap() error codes */
{ CE2_AES_WRAP_ILLEGAL_DATA_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_AES_WRAP_DATA_LENGTH_ERROR,				CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"DataLen must be divisible by 8 bytes and must be in range [16,  2^28]" },
{ CE2_AES_WRAP_KEY_LENGTH_ERROR,				CE2_KEY_SIZE_ERROR_MESSAGE
												"Supported key sizes are 128/192/256/512 bits." },
{ CE2_AES_WRAP_KEY_PTR_ERROR,					CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_AES_WRAP_ILLEGAL_WRAP_DATA_PTR_ERROR,		"Wrap data pointer is invalid" },
{ CE2_AES_WRAP_ILLEGAL_WRAP_DATA_LEN_PTR_ERROR,	"Wrap data length pointer is invalid" },
{ CE2_AES_WRAP_ILLEGAL_WRAP_DATA_LENGTH_ERROR,	"Wrap data length is invalid" },

/* CE2_AES_Unwrap() error codes */
{ CE2_AES_UNWRAP_ILLEGAL_WRAP_DATA_PTR_ERROR,	"Wrap data pointer is invalid" },
{ CE2_AES_UNWRAP_WRAP_DATA_LENGTH_ERROR,		"Wrap data length is invalid" },
{ CE2_AES_UNWRAP_KEY_LEN_ERROR,					CE2_KEY_SIZE_ERROR_MESSAGE
												"Supported 128/192/256/512 key sizes." },
{ CE2_AES_UNWRAP_KEY_PTR_ERROR,					CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_AES_UNWRAP_ILLEGAL_DATA_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_AES_UNWRAP_ILLEGAL_DATA_LEN_PTR_ERROR,	CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE },
{ CE2_AES_UNWRAP_ILLEGAL_DATA_LENGTH_ERROR,		CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
												"Data out length must be divisible by 8 bytes and must be not less than "
												"WrapDataInLen - CE2_AES_WRAP_BLOCK_SIZE_IN_BYTES." },

/* CE2_XTS_AES() error codes */
{ CE2_XTS_AES_ILLEGAL_TWEAK_PTR_ERROR,			"Tweak pointer is invalid: "
												"Pointer to buffer containing:\n"
												"- Tweak Value (if IsTweakInternal = FALSE) OR\n"
												"- Sector Number (if IsTweakInternal = TRUE)\n"
												"In case tweak value is generated internally it's returned through this parameter" },
{ CE2_XTS_AES_ILLEGAL_KEY_PTR_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_XTS_AES_ILLEGAL_KEY_SIZE_ERROR,			CE2_KEY_SIZE_ERROR_MESSAGE },
{ CE2_XTS_AES_ILLEGAL_ENCRYPT_MODE_ERROR,		CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE },
{ CE2_XTS_AES_ILLEGAL_DATAIN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_XTS_AES_TO_SHORT_DATA_SIZE_ERROR,			CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"It must be at least 16 bytes." },
{ CE2_XTS_AES_ILLEGAL_DATAOUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_XTS_AES_ILLEGAL_SECTOR_NUMBER_PTR_ERROR,	"Pointer to 64-bit sector number is invalid." },

/* CE2_AES_GCM() error codes */
{ CE2_AES_GCM_ILLEGAL_KEY_PTR_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_AES_GCM_KEY_SIZE_ERROR,					CE2_KEY_SIZE_ERROR_MESSAGE
												"Supported key sizes are 128/192/256 bits." },
{ CE2_AES_GCM_ILLEGAL_IV_PTR_ERROR,				"IV counter pointer is invalid" },
{ CE2_AES_GCM_ILLEGAL_AAD_PTR_ERROR,			"ADD pointer is invalid" },
{ CE2_AES_GCM_ENCRYPT_DECRYPT_FLAG_ERROR,		CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE },
{ CE2_AES_GCM_ILLEGAL_DATA_IN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_AES_GCM_ILLEGAL_DATA_OUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_AES_GCM_ILLEGAL_TAG_PTR_ERROR,			"Tag pointer is invalid" },
{ CE2_AES_GCM_ILLEGAL_TAG_SIZE_PTR_ERROR,		"Tag size pointer is invalid" }
}; /* End of AES_Errors */

/**
****************************************************************
* Data Name: 
* C2_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the C2 module.
***************************************************************/
struct ErrorItem C2_Errors[] =
{
/********************/
/* C2 module error  */
/********************/
/* C2 module on the CE2 layer base address - 0x00F01000 */
{ CE2_C2_MODULE_ERROR_BASE,						"C2 module on the CE2 layer base error" },

/* C2 module on the LLF layer base address -  0x00F11000 */
{ CE2_LLF_C2_MODULE_ERROR_BASE,					"C2 module on the LLF layer error" },

/* CE2_C2() error codes */
{ CE2_C2_CIPHER_INVALID_USER_CONTEXT_POINTER_ERROR, "" },
{ CE2_C2_CIPHER_ILLEGAL_OPERATION_MODE_ERROR,	"Operation mode is invalid: "
												"The operation mode of the C2Cipher (ECB or C-CBC)" },
{ CE2_C2_CIPHER_INVALID_KEY_POINTER_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_C2_CIPHER_INVALID_ENCRYPT_MODE_ERROR,	CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE },
{ CE2_C2_CIPHER_DATA_IN_POINTER_INVALID_ERROR, CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_C2_CIPHER_DATA_OUT_POINTER_INVALID_ERROR, CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_C2_CIPHER_DATA_SIZE_ILLEGAL,				CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"Must be divisible by 8 bytes." },
{ CE2_C2_CIPHER_DATA_OUT_DATA_IN_OVERLLAP_ERROR, "A partially overlap between the input and the output buffers." },

/* CE2_C2_HASH() error codes */
{ CE2_C2_HASH_DATA_IN_PTR_ERROR,				CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_C2_HASH_RESULT_BUFFER_PTR_ERROR,			CE2_DATA_OUT_PTR_ERROR_MESSAGE },

/* CE2_C2_OneWayFunc() error codes */
{ CE2_C2_ONEWAYFUNC_DATA1_PTR_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE
												"The pointer to the buffer of the input Data1 used as encrypting key."
												"The pointer need to be aligned to 32 bits." },
{ CE2_C2_ONEWAYFUNC_DATA2_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
												"The pointer to the 64-bits buffer of the input Data2."
												"The pointer need to be aligned to 32 bits." },
{ CE2_C2_ONEWAYFUNC_DATAOUT_PTR_ERROR,			CE2_DATA_OUT_PTR_ERROR_MESSAGE
												"The pointer to the 64-bits buffer for output data."
												"The pointer need to be aligned to 32 bits." },

/* LLF_C2_LoadSecretConstant() error codes */
{ CE2_LLF_C2_LOAD_SECTET_CONSTANT_CANT_OPEN_FILE, "Open secret constant file failed." },
{ CE2_LLF_C2_LOAD_SECTET_CONSTANT_CANT_READ_FILE, "Read secret constant file failed." }
}; /* End of C2_Errors */

/**
****************************************************************
* Data Name: 
* DES_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the DES module.
***************************************************************/
struct ErrorItem DES_Errors[] =
{
/********************/
/* DES module error */
/********************/
/* DES module on the CE2 layer base address - 0x00F00100 */
{ CE2_DES_MODULE_ERROR_BASE,					"DES module on the CE2 layer base error" },

/* DES module on the LLF layer base address -  0x00F10100 */
{ CE2_LLF_DES_MODULE_ERROR_BASE,				"DES module on the LLF layer base error" },

/* CE2_DES() error codes */
{ CE2_DES_INVALID_KEY_PTR_ERROR,				CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_DES_ILLEGAL_ENCRYPT_DECRYPT_FLAG_ERROR,	CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE },
{ CE2_DES_ILLEGAL_NUM_OF_KEYS_ERROR,			"Number of keys is invalid: "
												"The number of keys used: 1, 2, or 3." },
{ CE2_DES_ILLEGAL_HASH_OP_MODE_ERROR,			"Operation mode is invalid: "
												"Supported operation mode: ECB or CBC." },
{ CE2_DES_INVALID_DATA_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_DES_INVALID_DATA_IN_SIZE_ERROR,			CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"Must be grater than 0 and divisible by 8." }
}; /* End of DES_Errors */

/**
****************************************************************
* Data Name: 
* DH_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the DH module.
***************************************************************/
struct ErrorItem DH_Errors[] =
{
/********************/
/* DH module error  */
/********************/
/* DH module on the CE2 layer base address - 0x00F00500 */
{ CE2_DH_MODULE_ERROR_BASE,									"DH module on the CE2 layer base error" },

/* _DX_DH_GeneratePubPrv() error codes */
{ CE2_DH_GENERATE_PUB_PRV_GENARATOR_PTR_ERROR,				"Generator pointer is invalid" },
{ CE2_DH_GENERATE_PUB_PRV_PRIME_PTR_ERROR,					CE2_PRIME_PTR_ERROR_MESSAGE },
{ CE2_DH_GENERATE_PUB_PRV_Q_PTR_ERROR,						"Q pointer is invalid"
															"Relevant only for ANSI X9.42 -"
															"Pointer to the Q octet string 1 <= Prv <= q-1 or 1 < Prv < q-1" },
{ CE2_DH_GENERATE_PUB_PRV_ILLEGAL_DH_MODE,					"DH mode is invalid: "
															"Supported only PKCS#3 or ANSI X9.42"},
{ CE2_DH_GENERATE_PUB_PRV_CLIENT_PRIVATE_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"Client private key must be defined." },
{ CE2_DH_GENERATE_PUB_PRV_CLIENT_PRIVATE_KEY_SIZE_PTR_ERROR, CE2_KEY_SIZE_PTR_ERROR_MESSAGE
															"Client private key size must be defined."},
{ CE2_DH_GENERATE_PUB_PRV_CLIENT_PUBLIC_KEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"Client public key must be defined." },
{ CE2_DH_GENERATE_PUB_PRV_CLIENT_PUBLIC_KEY_SIZE_PTR_ERROR,	CE2_KEY_SIZE_PTR_ERROR_MESSAGE
															"Client public key size must be defined." },

/* _DX_DH_KeyDerivationFunc() error codes */
{ CE2_DH_KEY_DER_FUNC_ZZ_SECRET_PTR_ERROR,					"ZZSecret pointer is invalid." },
{ CE2_DH_KEY_DER_FUNC_KEYING_DATA_PTR_ERROR,				CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_DH_KEY_DER_FUNC_ILLEGAL_HASH_MODE_ERROR,				CE2_HASH_MODE_ERROR_MESSAGE
															"Supported only SHA1 mode." },

/* _DX_DH_X942_GetSecretData error codes */
{ CE2_DH_X942_GET_SECRET_DATA_ILLEGAL_HASH_MODE_ERROR,		CE2_HASH_MODE_ERROR_MESSAGE
															"Supported only SHA1 mode." },
{ CE2_DH_X942_GET_SECRET_DATA_CLIENT_PRIVATE_KEY_PTR_ERROR,	CE2_KEY_PTR_ERROR_MESSAGE
															"Client private key must be defined." },
{ CE2_DH_X942_GET_SECRET_DATA_PRIME_PTR_ERROR,				CE2_PRIME_PTR_ERROR_MESSAGE },
{ CE2_DH_X942_GET_SECRET_DATA_ILLEGAL_DER_FUNC_MODE_ERROR,	CE2_DER_FUNC_MODE_ERROR_MESSAGE },
{ CE2_DH_X942_GET_SECRET_DATA_SECRET_KEY_DATA_PTR_ERROR,	CE2_DATA_IN_PTR_ERROR_MESSAGE
															"Secret key must be defined." },

/* _DX_DH_X942_Hybrid_GetSecretData error codes */
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_ILLEGAL_HASH_MODE_ERROR,		CE2_HASH_MODE_ERROR_MESSAGE },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_CLIENT_PRIVATE_KEY_PTR1_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
																	"Client private key1 must be defined." },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_CLIENT_PRIVATE_KEY_PTR2_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
																	"Client private key2 must be defined." },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_SERVER_PUBLIC_KEY_PTR1_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
																	"Server public key1 must be defined." },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_SERVER_PUBLIC_KEY_PTR2_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
																	"Server public key2 must be defined." },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_PRIME_PTR_ERROR,				CE2_PRIME_PTR_ERROR_MESSAGE },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_ILLEGAL_DER_FUNC_MODE_ERROR,	CE2_DER_FUNC_MODE_ERROR_MESSAGE },
{ CE2_DH_X942_HYBRID_GET_SECRET_DATA_SECRET_KEY_DATA_PTR_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE
																	"Client private key1 must be defined."},

/* CE2_DH_PKCS3_GetSecretKey error codes */
{ CE2_DH_PKCS3_GET_SECRET_KEY_CLIENT_PRV_KEY_PTR_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE
															"Client private key must be defined." },
{ CE2_DH_PKCS3_GET_SECRET_KEY_SERVER_PUB_KEY_PTR_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE
															"Server public key must be defined." },
{ CE2_DH_PKCS3_GET_SECRET_KEY_PRIME_PTR_ERROR,				CE2_PRIME_PTR_ERROR_MESSAGE },
{ CE2_DH_PKCS3_GET_SECRET_KEY_SECRET_KEY_PTR_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE
															"A pointer to the secret key octet string must be defined. " },
{ CE2_DH_PKCS3_GET_SECRET_KEY_SECRET_KEY_SIZE_PTR_ERROR,	CE2_KEY_SIZE_PTR_ERROR_MESSAGE
															"A pointer to the secret key size must be defined. "},

/* CE2_DH_CheckPubKey error codes */
{ CE2_DH_INVALID_ARGUMENT_POINTER_ERROR,					CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_DH_INVALID_MODULUS_SIZE_ERROR,						CE2_PRIME_PTR_ERROR_MESSAGE
															"Mudulus size must be between 1024 and 2048 bytes inclusive "
															"and divisible by 256." },
{ CE2_DH_INVALID_ORDER_SIZE_ERROR,							CE2_DATA_IN_SIZE_ERROR_MESSAGE
															"Q size must be between 160 and modulus size bits inclusive "
															"and divisible by 32 bits." },
{ CE2_DH_INVALID_PUBLIC_KEY_SIZE_ERROR,						CE2_KEY_SIZE_ERROR_MESSAGE
															"Public key size must be less than prime size." },
{ CE2_DH_INVALID_PUBLIC_KEY_ERROR,							"Public key is invalid." },

/* CE2_DH_CheckDomainParams error codes */
{ CE2_DH_CHECK_SEED_SIZE_OR_PTR_NOT_VALID_ERROR,			"Q pointer or size is invalid." },
{ CE2_DH_CHECK_GENERATOR_SIZE_OR_PTR_NOT_VALID_ERROR,		"Generator pointer or size is invalid." },
{ CE2_DH_CHECK_DOMAIN_PRIMES_NOT_VALID_ERROR,				"Domain primes are not valid." },
{ CE2_DH_CHECK_GENERATOR_NOT_VALID_ERROR,					"Generator is not valid." },

/* CE2_DH_CreateDomainParams error codes */
{ CE2_DH_INVALID_SEED_SIZE_ERROR,							"Seed size is invalid. "
															"Seed size must be between 160 and modulus size / 2 bits inclusive." },
{ CE2_DH_INVALID_GENERATOR_PTR_OR_SIZE_ERROR,				"Generator pointer or size is invalid." },
{ CE2_DH_INVALID_J_FACTOR_PTR_OR_SIZE_ERROR,				"J factor pointer or size is invalid." },

}; /* End of DH_Errors */

/**
****************************************************************
* Data Name: 
* ECPKI_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the ECPKI module.
***************************************************************/
struct ErrorItem ECPKI_Errors[] =
{
/**********************/
/* ECPKI module error */
/**********************/
/* ECPKI module on the CE2 layer base address - 0x00F00800 */
{ CE2_ECPKI_MODULE_ERROR_BASE,								"ECPKI module on the CE2 layer base error" },

/* ECPKI module on the LLF layer base address -  0x00F10800 */
{ CE2_LLF_ECPKI_MODULE_ERROR_BASE,							"ECPKI module on the LLF layer base error" },

/* CE2_ECPKI_GenKeyPair() error codes */
{ CE2_ECPKI_GEN_KEY_ILLEGAL_DOMAIN_ID_ERROR,				CE2_DOMAIN_ID_ERROR_MESSAGE },
{ CE2_ECPKI_GEN_KEY_INVALID_PRIVATE_KEY_PTR_ERROR,			CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"Private key must be defined." },
{ CE2_ECPKI_GEN_KEY_INVALID_PUBLIC_KEY_PTR_ERROR,			CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"Public key must be defined." },

/* CE2_ECPKI_BuildPrivKey() error codes */
{ CE2_ECPKI_BLD_PRIV_KEY_ILLEGAL_DOMAIN_ID_ERROR,			CE2_DOMAIN_ID_ERROR_MESSAGE },
{ CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_IN_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"This structure is used as input for the ECPKI cryptographic primitives. "
															"Size of buffer must be not less than: "
															"Order size of the appropriative domain in bytes" },
/* LLF_ECPKI_BuildPrivKey() error codes */
{ LLF_ECPKI_BLD_PRIV_KEY_INVALID_PRIVATE_KEY_SIZE_ERROR,	CE2_KEY_SIZE_ERROR_MESSAGE
															"Imported private key size is invalid." },

/* CE2_ECPKI_BuildPublKey() error codes */
{ CE2_ECPKI_BLD_PUBL_KEY_ILLEGAL_DOMAIN_ID_ERROR,			CE2_DOMAIN_ID_ERROR_MESSAGE },
{ CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_IN_PTR_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"This structure is used as input for the ECPKI cryptographic primitives. "
															"Size of buffer must be not less than: "
															"2*ModSizeInBytes - for WM DRM, 2*ModSizeInBytes+1 - for other domains." },
/* LLF_ECPKI_BuildPublKey() error codes */
{ LLF_ECPKI_BLD_PUBL_KEY_INVALID_PUBLIC_KEY_SIZE_ERROR,		CE2_KEY_SIZE_ERROR_MESSAGE
															"Imported public key size is invalid." },

/* CE2_ECPKI_SVDP_DH() error codes */
{ CE2_ECPKI_SVDP_DH_INVALID_BUILDED_PARTNER_PUBLIC_KEY_PTR_ERROR, CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"Partner public key must be defined." },
{ CE2_ECPKI_SVDP_DH_INVALID_USER_PRIVATE_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"User private key must be defined." },
{ CE2_ECPKI_SVDP_DH_INVALID_SHARED_SECRET_VALUE_PTR_ERROR,	"Shared secret value pointer is invalid: "
															"A pointer to an output buffer that will contain the shared secret value." },
{ CE2_ECPKI_SVDP_DH_INVALID_SHARED_SECRET_VALUE_SIZE_PTR_ERROR,"Shared secret value size pointer is invalid: "
															"A pointer to the size of user passed buffer (in) and actual output "
															"size of calculated shared secret value." },
{ CE2_ECPKI_SVDP_DH_INVALID_DOMAIN_ID_ERROR,				CE2_DOMAIN_ID_ERROR_MESSAGE },

/* LLF_ECPKI_SVDP_DH() error codes */
{ LLF_ECPKI_SVDP_DH_INVALID_SHARED_SECRET_VALUE_SIZE_ERROR,	"Shared secret value size is invalid: "
															"Output buffer for shared secret value is too small. "
															"Pass large buffer to exclude error." },

/* CE2_ECDSA_Sign() error codes */
{ CE2_ECDSA_SIGN_INVALID_USER_PRIV_KEY_PTR_ERROR,			CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"Signer private key must be defined." },
{ CE2_ECDSA_SIGN_ILLEGAL_HASH_OP_MODE_ERROR,				CE2_HASH_MODE_ERROR_MESSAGE
															"Supported modes are SHA1/224/256/384/512 and modes after SHA1/224/256/384/512." },
{ CE2_ECDSA_SIGN_INVALID_MESSAGE_DATA_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
															"Input message must be defined." },
{ CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_PTR_ERROR,			CE2_DATA_OUT_PTR_ERROR_MESSAGE
															"Signature output buffer must be defined." },
{ CE2_ECDSA_SIGN_INVALID_SIGNATURE_OUT_SIZE_PTR_ERROR,		CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
															"Signature output size pointer must be defined. "
															"It contains size of user passed buffer for signature (in) and size of actual signature (out)." },
{ CE2_ECDSA_SIGN_INVALID_EPHEMERAL_KEY_IN_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE
															"Ephimeral private key must be defined." },

/* CE2_ECDSA_Verify() error codes */
{ CE2_ECDSA_VERIFY_INVALID_SIGNER_PUBL_KEY_PTR_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_ECDSA_VERIFY_ILLEGAL_HASH_OP_MODE_ERROR,				CE2_HASH_MODE_ERROR_MESSAGE },
{ CE2_ECDSA_VERIFY_INVALID_SIGNATURE_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
															"Input signature data must be defined." },
{ CE2_ECDSA_VERIFY_INVALID_MESSAGE_DATA_IN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE
															"A pointer to message to be verified must be defined." },

/* _DX_ECPKI_ELGAMAL_Encrypt() error codes */
{ CE2_EC_ELGAMAL_ENCR_INVALID_RECEIVER_PUBL_KEY_PTR_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"A pointer to receiver public key (key in little endian) must be defined." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_MESSAGE_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
															"A pointer to message to be encrypted (message in big endian) must be defined." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_MESSAGE_OUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE
															"A pointer to buffer for encrypted message must be defined." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_MESSAGE_OUT_SIZE_PTR_ERROR,	CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
															"A pointer to output message size must be defined." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_EPHEMERAL_KEY_IN_ERROR,		CE2_KEY_PTR_ERROR_MESSAGE
															"Ephimeral private key must be defined." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_MESSAGE_IN_SIZE_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE
															"A size of incoming message in bytes is invalid. "
															"The size must be MessageInSizeBytes =< ModulusSizeInBytes -  1. "
															"Note: in case CE2_ECPKI_DomainID_secp521r1 MessageInSizeBytes =< ModulusSizeInBytes -  2." },
{ CE2_EC_ELGAMAL_ENCR_INVALID_MESSAGE_OUT_SIZE_ERROR,		CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
															"Output buffer for message is too small. "
															"Pass large buffer to exclude error." },

/* CE2_ECPKI_ELGAMAL_Decrypt */
{ CE2_EC_ELGAMAL_DECR_INVALID_RECEIVER_PRIV_KEY_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
															"A pointer to a receiver private key structure (in affine coordinates) must be defined." },
{ CE2_EC_ELGAMAL_DECR_INVALID_ENCR_MESSAGE_IN_PTR_ERROR,	CE2_DATA_IN_PTR_ERROR_MESSAGE
															"The encrypted message buffer must be defined." },
{ CE2_EC_ELGAMAL_DECR_INVALID_DECR_MESSAGE_OUT_PTR_ERROR,	CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
															"The decrypted message buffer must be defined." },
{ CE2_EC_ELGAMAL_DECR_INVALID_DECR_MESSAGE_OUT_SIZE_PTR_ERROR,	CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
															"Pointer to buffer to output the size of decrypted message." },
{ CE2_EC_ELGAMAL_DECR_INVALID_ENCR_MESSAGE_IN_SIZE_ERROR,	CE2_DATA_IN_PTR_ERROR_MESSAGE
															"Input message buffer is too small. "
															"Pass large buffer to exclude error." },
{ CE2_EC_ELGAMAL_DECR_INVALID_ENCR_MESSAGE_OUT_SIZE_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE
															"Output message buffer is too small. "
															"Pass large buffer to exclude error." },

/* CE2_ECPKI_ExportPublKey error codes */
{ CE2_ECPKI_EXPORT_PUBL_KEY_INVALID_USER_PUBL_KEY_PTR_ERROR,CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"A pointer to the public key structure (in little-endian form) must be defined." },
{ CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_COMPRESSION_MODE_ERROR,	"Point compression mode is invalid: "
															"Supported only Compressed, Uncompressed or Hybrid compression." },
{ CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
															"A pointer to the exported public key structure (in big-endian) must be defined. "
															"Size of buffer must be not less than: "
															"2*ModSizeInBytes - for WM DRM, 2*ModSizeInBytes+1 - for other domains. " },
{ CE2_ECPKI_EXPORT_PUBL_KEY_ILLEGAL_PUBL_KEY_SIZE_PTR_ERROR,CE2_KEY_SIZE_ERROR_MESSAGE
															"Public key size must be defined." },
{ LLF_ECPKI_EXPORT_PUBL_KEY_INVALID_PUBL_KEY_SIZE_ERROR,	CE2_KEY_SIZE_ERROR_MESSAGE
															"Output buffer for public key value is too small. "
															"Pass large buffer to exclude error." },

/* CE2_ECPKI_ExportPrivKey error codes */
{ CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_USER_PRIV_KEY_PTR_ERROR,			CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
																		"A pointer to the user private key structure (in little-endian form)." },
{ CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
																		"A pointer to the exported private key (in big-endian) must be defined. "
																		"Size of buffer must be not less than: "
																		"Order size of the appropriative domain in bytes." },
{ CE2_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_SIZE_PTR_ERROR, CE2_KEY_SIZE_ERROR_MESSAGE
																		"A pointer to a variable for input size of user passed buffer for "
																		"private key and output the size of converted private key in bytes "
																		"must be defined." },
/* LLF_ECPKI_ExportPrivKey error codes */
{ LLF_ECPKI_EXPORT_PRIVATE_KEY_INVALID_EXTERN_PRIVATE_KEY_SIZE_ERROR,	CE2_KEY_SIZE_ERROR_MESSAGE
																		"Output key size buffer is too small. "
																		"Pass large buffer to exclude error." }
}; /* End of ECPKI_Errors */

/**
****************************************************************
* Data Name: 
* HASH_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the HASH module.
***************************************************************/
struct ErrorItem HASH_Errors[] =
{
/**********************/
/* HASH module error  */
/**********************/
/* HASH module on the CE2 layer base address - 0x00F00200 */
{ CE2_HASH_MODULE_ERROR_BASE,					"HASH module on the CE2 layer base error" },

/* HASH module on the LLF layer base address -  0x00F10200 */
{ CE2_LLF_HASH_MODULE_ERROR_BASE,				"HASH module on the LLF layer base error" },

/* CE2_HASH() error codes */
{ CE2_HASH_ILLEGAL_HASH_OP_MODE_ERROR,			CE2_HASH_MODE_ERROR_MESSAGE
												"Supported modes are SHA1/224/256/384/512 and MD5." },
{ CE2_HASH_INVALID_DATA_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
												"A pointer to the buffer that stores data to be hashed must be defined." },
{ CE2_HASH_INVALID_HASH_RESULT_BUFF_PTR_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE
												"A pointer to the target buffer where the HASH result "
												"stored in the context is loaded to must be defined" }
}; /* End of HASH_Errors */

/**
****************************************************************
* Data Name: 
* HMAC_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the HMAC module.
***************************************************************/
struct ErrorItem HMAC_Errors[] =
{
/**********************/
/* HMAC module error  */
/**********************/
/* HMAC module on the CE2 layer base address - 0x00F00300 */
{ CE2_HMAC_MODULE_ERROR_BASE,					"HMAC module on the CE2 layer base error" },

/* HMAC module on the LLF layer base address - 0x00F10300 */
{ CE2_LLF_HMAC_MODULE_ERROR_BASE,				"HMAC module on the LLF layer base error" },

/* CE2_HMAC() error codes */
{ CE2_HMAC_ILLEGAL_HASH_OP_MODE_ERROR,			CE2_HASH_MODE_ERROR_MESSAGE
												"Supported modes are SHA1/224/256/384/512 and MD5." },
{ CE2_HMAC_INVALID_KEY_PTR_ERROR,				CE2_KEY_PTR_ERROR_MESSAGE
												"A pointer to the user key buffer must be defined." },
{ CE2_HMAC_INVALID_DATA_IN_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE
												"A pointer to the buffer that stores the data to be hashed must be defined." },
{ CE2_HMAC_INVALID_HASH_RESULT_BUFF_PTR_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE
												"A pointer to the target buffer where the HMAC result "
												"stored in the context is loaded to must be defined" }
}; /* End of HMAC_Errors */

/**
****************************************************************
* Data Name: 
* RC4_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the RC4 module.
***************************************************************/
struct ErrorItem RC4_Errors[] =
{
/*********************/
/* RC4 module error  */
/*********************/
/* RC4 module on the CE2 layer base address - 0x00F00600 */
{ CE2_RC4_MODULE_ERROR_BASE,					"RC4 module on the CE2 layer base error" },

/* CE2_RC4() error codes */
{ CE2_RC4_KEY_PTR_ERROR,						CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_RC4_DATA_IN_PTR_ERROR,					CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RC4_DATA_OUT_PTR_ERROR,					CE2_DATA_OUT_PTR_ERROR_MESSAGE
												"The pointer to the buffer of the output data from the RC4 must be defined. "
												"The pointer's value does not need to be word-aligned. "
												"The size of this buffer must be the same as the DataIn buffer." }
}; /* End of RC4_Errors */

/**
****************************************************************
* Data Name: 
* RSA_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the RSA module.
***************************************************************/
struct ErrorItem RSA_Errors[] =
{
/*********************/
/* RSA module error  */
/*********************/
/* PKI RSA module on the CE2 layer base address - 0x00F00400 */
{ CE2_RSA_MODULE_ERROR_BASE,							"PKI RSA module on the CE2 layer base error" },

/* PKI RSA module on the LLF layer base address - 0x00F10400 */
{ CE2_LLF_RSA_MODULE_ERROR_BASE,						"PKI RSA module on the LLF layer base error" },

/* CE2_RSA_Build_PubKey() error code */
{ CE2_RSA_BUILD_PUBKEY_INVALID_PUB_KEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PUBKEY_INVALID_EXPONENT_PTR_ERROR,		CE2_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_PTR_ERROR,		CE2_MODULUS_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_MS_BIT_ERROR,	CE2_MODULUS_MS_BIT_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PUBKEY_INVALID_MODULUS_SIZE,			CE2_MODULUS_SIZE_ERROR_MESSAGE },

/* CE2_RSA_Build_PrivKey() error code */
{ CE2_RSA_BUILD_PRIVKEY_INVALID_PRIV_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEY_INVALID_PRIV_EXPONENT_PTR_ERROR,CE2_PRV_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEY_INVALID_PUB_EXPONENT_PTR_ERROR,	CE2_PUB_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_PTR_ERROR,		CE2_MODULUS_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_MS_BIT_ERROR,	CE2_MODULUS_MS_BIT_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEY_INVALID_MODULUS_SIZE,			CE2_MODULUS_SIZE_ERROR_MESSAGE },

/* CE2_RSA_Build_PrivKeyCRT() error code */
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_PRIV_KEY_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_P_PTR_ERROR,			"P value must be defined." },
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_Q_PTR_ERROR,			"Q value must be defined." },
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_DP_PTR_ERROR,		"dP value must be defined." },
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_DQ_PTR_ERROR,		"dQ value must be defined." },
{ CE2_RSA_BUILD_PRIVKEYCRT_INVALID_QINV_PTR_ERROR,		"qInv value must be defined." },

/* CE2_RSA_Get_PubKey() error code */
{ CE2_RSA_GET_PUBKEY_INVALID_PUB_KEY_PTR_ERROR,			CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PUBKEY_INVALID_EXPONENT_PTR_ERROR,		CE2_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PUBKEY_INVALID_MODULUS_PTR_ERROR,			CE2_MODULUS_PTR_ERROR_MESSAGE },

/* CE2_RSA_Get_ModSizeFromPubKey() error code */
{ CE2_RSA_GET_MODSIZEFROMPUBKEY_INVALID_PUB_KEY_PTR_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_MODSIZEFROMPUBKEY_INVALID_MODULUS_PTR_ERROR,	CE2_MODULUS_PTR_ERROR_MESSAGE },

/* CE2_RSA_Get_PrivKey() error code */
{ CE2_RSA_GET_PRIVKEY_INVALID_PRIV_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PRIVKEY_INVALID_PRIV_EXPONENT_PTR_ERROR,	CE2_PRV_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PRIVKEY_INVALID_PRIV_EXPONENT_SIZE_PTR_ERROR,	CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Private exponent input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEY_INVALID_PUB_EXPONENT_PTR_ERROR,	CE2_PUB_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PRIVKEY_INVALID_PUB_EXPONENT_SIZE_PTR_ERROR,	CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Public exponent input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEY_INVALID_MODULUS_PTR_ERROR,		CE2_MODULUS_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PRIVKEY_INVALID_MODULUS_PTR_SIZE_ERROR,	CE2_MODULUS_SIZE_ERROR_MESSAGE },

/* CE2_RSA_Get_PrivKeyCRT() error code */
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_PRIV_KEY_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_P_PTR_ERROR,			"P value must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_P_SIZE_PTR_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"P input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_Q_PTR_ERROR,			"Q value must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_Q_SIZE_PTR_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Q input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_DP_PTR_ERROR,			"dP value must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_DP_SIZE_PTR_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"dP input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_DQ_PTR_ERROR,			"dQ value must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_DQ_SIZE_PTR_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"dQ input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_QINV_PTR_ERROR,		"qInv value must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_INVALID_QINV_SIZE_PTR_ERROR,	CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"qInv input/output size must be defined." },
{ CE2_RSA_GET_PRIVKEYCRT_IS_NOT_CRT_KEY_ERROR,			"Private ke is not in CRT mode: "
														"To retrieve P, Q, dP, dQ and dInv values private key must be in CRT mode." },

/* CE2_RSA_KG_GenerateKeyPair() error code */
{ CE2_RSA_KG_GENKEYPAIR_INVALID_PUBEXP_PTR_ERROR,		CE2_PUB_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIR_INVALID_PUBKEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIR_INVALID_PRIVKEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIR_INVALID_MODULUS_SIZE_ERROR,		CE2_MODULUS_SIZE_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIR_INVALID_EXPONENT_SIZE_ERROR,	CE2_EXPONENT_SIZE_ERROR_MESSAGE },

/* CE2_RSA_KG_GenerateKeyPairCRT() error code */
{ CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PUBEXP_PTR_ERROR,	CE2_PUB_EXPONENT_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PUBKEY_PTR_ERROR,	CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIRCRT_INVALID_PRIVKEY_PTR_ERROR,	CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIRCRT_INVALID_MODULUS_SIZE_ERROR,	CE2_MODULUS_SIZE_ERROR_MESSAGE },
{ CE2_RSA_KG_GENKEYPAIRCRT_INVALID_EXPONENT_SIZE_ERROR,	CE2_EXPONENT_SIZE_ERROR_MESSAGE },

/* CE2_RSA_PRIM_Encrypt() error code */
{ CE2_RSA_PRIM_ENCRYPT_INVALID_PUBKEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_PRIM_ENCRYPT_INVALID_DATA_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RSA_PRIM_ENCRYPT_INVALID_OUTPUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },

/* CE2_RSA_PRIM_Decrypt() error code */
{ CE2_RSA_PRIM_DECRYPT_INVALID_PUBKEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_PRIM_DECRYPT_INVALID_DATA_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RSA_PRIM_DECRYPT_INVALID_OUTPUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },

/* _DX_RSA_SCHEMES_Encrypt() error code */
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_PUBKEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_HASHMODE_ERROR,		CE2_RSA_HASH_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_MGF_MODE_ERROR,		CE2_MGF_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_PKCS_VERSION_ERROR,	CE2_PKCS_VERSION_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_DATAIN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_ENCRYPT_INVALID_DATAOUT_PTR_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE },

/* _DX_RSA_SCHEMES_Decrypt() error code */
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_DATA_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_DATASIZE_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Must be <= the size of the modulus." },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_OUTPUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_OUTPUT_SIZE_PTR_ERROR,CE2_DATA_OUT_SIZE_PTR_ERROR_MESSAGE
														"Must be >= the size of the modulus." },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_HASH_MODE_ERROR,		CE2_RSA_HASH_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_MGF_MODE_ERROR,		CE2_MGF_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_DECRYPT_INVALID_PKCS_MODE_ERROR,		CE2_PKCS_VERSION_ERROR_MESSAGE },

/* _DX_RSA_Sign() error code */
{ CE2_RSA_SCHEMES_SIGN_INVALID_PRIVKEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_SIGN_PRIVKEY_VALIDATION_TAG_ERROR,	"Private key validation tag is invalid: "
														"It's possible that key structure version is "
														"invalid or key is corrupted." },
{ CE2_RSA_SCHEMES_SIGN_ILLEGAL_HASH_MODE_ERROR,			CE2_RSA_HASH_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_SIGN_ILLEGAL_MGF_ERROR,				CE2_MGF_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_SIGN_ILLEGAL_PKCS_VERSION_ERROR,		CE2_PKCS_VERSION_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_SIGN_INVALID_DATA_PTR_ERROR,			CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_SIGN_INVALID_OUTPUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },

/* _DX_RSA_Verify() error code */
{ CE2_RSA_SCHEMES_VERIFY_INVALID_PUBKEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_VERIFY_PUBKEY_VALIDATION_TAG_ERROR,	"Public key validation tag is invalid: "
														"It's possible that key structure version is "
														"invalid or key is corrupted." },
{ CE2_RSA_SCHEMES_VERIFY_ILLEGAL_HASH_MODE_ERROR,		CE2_RSA_HASH_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_VERIFY_ILLEGAL_MGF_ERROR,				CE2_MGF_MODE_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_VERIFY_ILLEGAL_PKCS_VERSION_ERROR,	CE2_PKCS_VERSION_ERROR_MESSAGE },
{ CE2_RSA_SCHEMES_VERIFY_INVALID_DATAIN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE }
}; /* End of RSA_Errors */

/**
****************************************************************
* Data Name: 
* OTF_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the OTF module.
***************************************************************/
struct ErrorItem OTF_Errors[] =
{
/*********************/
/* OTF module error  */
/*********************/
/* OTF module on the CE2 layer base address - 0x00F01100 */
{ CE2_OTF_MODULE_ERROR_BASE,							"OTF module on the CE2 layer base error" },

/* OTF module on the LLF layer base address -  0x00F11100 */
{ CE2_LLF_OTF_MODULE_ERROR_BASE,						"OTF module on the LLF layer base error" },

/* DX_AES2AES2HASH_OTF error code */
{ CE2_OTF_AES2AES2HASH_INVALID_DATAIN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_OTF_AES2AES2HASH_INVALID_DATAOUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_OTF_AES2AES2HASH_ILLEGAL_HASH_CHAIN_MODE_ERROR,	CE2_OTF_HASH_CHAIN_MODE_ERROR_MESSAGE },

/* DX_AES2DES2HASH_OTF error code */
{ CE2_OTF_AES2DES2HASH_INVALID_DATAIN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_OTF_AES2DES2HASH_INVALID_DATAOUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_OTF_AES2DES2HASH_ILLEGAL_HASH_CHAIN_MODE_ERROR,	CE2_OTF_HASH_CHAIN_MODE_ERROR_MESSAGE },

/* DX_DES2AES2HASH_OTF error code */
{ CE2_OTF_DES2AES2HASH_INVALID_DATAIN_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_OTF_DES2AES2HASH_INVALID_DATAOUT_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_OTF_DES2AES2HASH_ILLEGAL_HASH_CHAIN_MODE_ERROR,	CE2_OTF_HASH_CHAIN_MODE_ERROR_MESSAGE }
}; /* End of OTF_Errors */

/**
****************************************************************
* Data Name: 
* KDF_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the KDF module.
***************************************************************/
struct ErrorItem KDF_Errors[] =
{
/*********************/
/* KDF module error  */
/*********************/
/* KDF module on the CE2 layer base address - 0x00F01200 */
{ CE2_KDF_MODULE_ERROR_BASE,					"KDF module on the CE2 layer base error" },

/* KDF module on the LLF layer base address -  0x00F11200 */
{ CE2_LLF_KDF_MODULE_ERROR_BASE,				"KDF module on the LLF layer base error" },

/* _DX_KDF_KeyDerivFunc error code */
{ CE2_KDF_KEY_DERIV_FUNC_ZZSECRET_PTR_ERROR,	"ZZSecret pointer is invalid: "
												"A pointer to shared secret value octet string must be defined." },
{ CE2_KDF_KEY_DERIV_FUNC_OTHERINFO_PTR_ERROR,	"OtherInfo pointer is invalid: "
												"A pointer to structure, containing pointers and sizes of optional "
												"data shared by two entities intended to share the secret value. "
												"This argument (structure) and also its members are optional "
												"(if any is not needed - set its pointer and size to NULL)." },
{ CE2_KDF_KEY_DERIV_FUNC_KEYINGDATA_PTR_ERROR,	"Keying data pointer is invalid: "
												"A pointer to the keying data derived from the secret key, "
												"of length KeyLenInBits must be defined." },
{ CE2_KDF_KEY_DERIV_FUNC_KDF_HASH_MODE_ERROR,	"Hash mode is invalid: "
												"Supported are only SHA1/224/256/384/512 modes. "
												"The hash function output must be at least 160 bits." },
{ CE2_KDF_KEY_DERIV_FUNC_DERIVATION_MODE_ERROR,	"Function derivation mode is invalid: "
												"Specifies one of above described derivation modes: ASN1, OMADRM." }
}; /* End of KDF_Errors */

/**
****************************************************************
* Data Name: 
* RND_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the RND module.
***************************************************************/
struct ErrorItem RND_Errors[] =
{
/*********************/
/* RND module error  */
/*********************/
/* RND module on the CE2 layer base address - 0x00F00C00 */
//{ CE2_RND_MODULE_ERROR_BASE,							"RND module on the CE2 layer base error" },
/* RND module on the LLF layer base address -  0x00F10C00 */
{ CE2_LLF_RND_MODULE_ERROR_BASE,						"RND module on the LLF layer base error" },

{ CE2_RND_DATA_OUT_POINTER_INVALID_ERROR,				CE2_DATA_OUT_PTR_ERROR_MESSAGE
														"Output vector buffer must be defined." },
{ CE2_RND_CAN_NOT_GENERATE_RAND_IN_RANGE,				"Can not generate rand in range." },
{ CE2_RND_CPRNG_TEST_FAIL_ERROR,						"CPRNG test fail." },
{ CE2_RND_ADDITIONAL_INPUT_BUFFER_NULL,					CE2_DATA_IN_PTR_ERROR_MESSAGE
														"Additional input buffer ust be defined." },
{ CE2_RND_ADDITIONAL_INPUT_SIZE_ERROR,					CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Additional input buffer size must be 128 or 256 bits only." },

/* CE2_RND_GenerateVector error code */
{ CE2_RND_GENERATE_VECTOR_OUTPUT_PTR_ERROR,				CE2_DATA_OUT_PTR_ERROR_MESSAGE
														"Output vector buffer must be defined." },

/* CE2_RND_GenerateVectorInRange error code */
{ CE2_RND_GENERATE_VECTOR_IN_RANGE_MAXVECT_PTR_ERROR,	CE2_DATA_IN_PTR_ERROR_MESSAGE
														"The pointer to vector defines high limit of random "
														"vector must be defined." },
{ CE2_RND_GENERATE_VECTOR_IN_RANGE_RNDVECT_PTR_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE
														"Output vector buffer must be defined."}
}; /* End of RND_Errors */

/**
****************************************************************
* Data Name: 
* CMLA_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the CMLA module.
***************************************************************/
struct ErrorItem CMLA_Errors[] =
{
/**********************/
/* CMLA module error  */
/**********************/
/* CMLA module on the CE2 layer base address - 0x00F00900 */
{ CE2_CMLA_MODULE_ERROR_BASE,							"CMLA module on the CE2 layer base error" },

/* CMLA module on the LLF layer base address - 0x00F10900 */
{ CE2_LLF_CMLA_MODULE_ERROR_BASE,						"CMLA module on the LLF layer base error" },

/* CE2_CMLA_KDF error code */
{ CE2_CMLA_KDF_INVALID_SEED_POINTER_ERROR,				"X pointer is invalid: "
														"Seed value, an octet string of length 128 must be defined." },
{ CE2_CMLA_KDF_INVALID_KEK_POINTER_ERROR,				CE2_KEK_PTR_ERROR_MESSAGE },

/* CE2_CMLA_Wrap error code */
{ CE2_CMLA_WRAP_ILLEGAL_KEK_PTR_ERROR,					CE2_KEK_PTR_ERROR_MESSAGE },
{ CE2_CMLA_WRAP_ILLEGAL_KEY_PTR_ERROR,					CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_CMLA_WRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR,			CE2_WRAPPED_PTR_ERROR_MESSAGE },

/* CE2_CMLA_Unwrap error code */
{ CE2_CMLA_UNWRAP_ILLEGAL_KEK_PTR_ERROR,				CE2_KEK_PTR_ERROR_MESSAGE },
{ CE2_CMLA_UNWRAP_ILLEGAL_KEY_PTR_ERROR,				CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_CMLA_UNWRAP_ILLEGAL_WRAPPED_KEY_PTR_ERROR,		CE2_WRAPPED_PTR_ERROR_MESSAGE },

/* CE2_CMLA_RSA_Encrypt error code */
{ CE2_CMLA_RSA_ENCRYPT_ILLEGAL_PUB_KEY_PTR_ERROR,		CE2_PUBLIC_KEY_PTR_ERROR_MESSAGE
														"RSA public key must be defined." },
{ CE2_CMLA_RSA_ENCRYPT_ILLEGAL_MESSAGE_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE
														"A pointer to message of length 128 bytes must be defined." },
{ CE2_CMLA_RSA_ENCRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE
														"A pointer to output encrypted message of length 128 bytes must be defined." },
{ CE2_CMLA_RSA_ENCRYPT_MESSAGE_INTEGER_TOO_LONG_ERROR,	CE2_DATA_IN_SIZE_ERROR_MESSAGE
														"Input message size must be < modulus size." },

/* CE2_CMLA_RSA_Decrypt error code */
{ CE2_CMLA_RSA_DECRYPT_ILLEGAL_PRIV_KEY_PTR_ERROR,		CE2_PRIVATE_KEY_PTR_ERROR_MESSAGE
														"RSA private key must be defined." },
{ CE2_CMLA_RSA_DECRYPT_ILLEGAL_MESSAGE_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE
														"A pointer to output decrypted message must be defined." },
{ CE2_CMLA_RSA_DECRYPT_ILLEGAL_ENCR_MESSAGE_PTR_ERROR,	CE2_DATA_IN_PTR_ERROR_MESSAGE
														"A pointer to input encrypted message must be defined." }
}; /* End of CMLA_Errors */

/**
****************************************************************
* Data Name: 
* UTILS_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the UTILS module.
***************************************************************/
struct ErrorItem UTILS_Errors[] =
{
/**********************/
/* Utils module error */
/**********************/
/* UTILS module on the CE2 layer base address - 0x00F01300 */
{ CE2_UTILS_MODULE_ERROR_BASE,					"UTILS module on the CE2 layer base error" },
/* UTILS module on the LLF layer base address -  0x00F11300 */
{ CE2_LLF_UTILS_MODULE_ERROR_BASE,				"UTILS module on the LLF layer base error" },
{ CE2_UTILS_INVALID_POINTER_ERROR,				"Invalid pointer." }
}; /* End of UTILS_Errors */

/**
****************************************************************
* Data Name: 
* Base64_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the Base64 module.
***************************************************************/
struct ErrorItem Base64_Errors[] =
{
/**********************/
/* Base64 module error */
/**********************/
/* BASE64 module on the CE2 layer base address - 0x00F01400 */
{ CE2_BASE64_MODULE_ERROR_BASE,						"BASE64 module on the CE2 layer base error" },
/* BASE64 module on the LLF layer base address -  0x00F11400 */
{ CE2_LLF_BASE64_MODULE_ERROR_BASE,					"UTILS module on the LLF layer base error" },

{ CE2_BASE64_INPUT_BUFFER_INVALID_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_BASE64_OUTPUT_BUFFER_INVALID_PTR_ERROR,		CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_BASE64_OUTPUT_BUFFER_SIZE_INVALID_PTR_ERROR,	CE2_DATA_OUT_SIZE_ERROR_MESSAGE },
{ CE2_BASE64_OUTPUT_BUFFER_SIZE_TOO_SMALL_ERROR,	CE2_DATA_OUT_SIZE_ERROR_MESSAGE
													"Output buffer too small. Provide larger output buffer." },
}; /* End of Base64_Errors */

/**
****************************************************************
* Data Name: 
* INTERNAL_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the INTERNAL module.
***************************************************************/
struct ErrorItem INTERNAL_Errors[] =
{
/* INTERNAL debug error module on the CE2 layer base address - 0x00F00700 */
{ CE2_INTERNAL_MODULE_ERROR_BASE,				"INTERNAL debug error module on the CE2 layer base error" },
}; /* End of INTERNAL_Errors */

/**
****************************************************************
* Data Name: 
* SST_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the SST module.
***************************************************************/
struct ErrorItem SST_Errors[] =
{
/* CRY_SST module on the CE2 layer base address - 0x00F00A00 */
{ CE2_SST_MODULE_ERROR_BASE,					"CRY_SST module on the CE2 layer base error" },
}; /* End of CRY_SST_Errors */

/**
****************************************************************
* Data Name: 
* PKI_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the PKI module.
***************************************************************/
struct ErrorItem PKI_Errors[] =
{
/* PKI RSA module on the LLF layer base address -  0x00F10B00 */
{ CE2_LLF_PKI_MODULE_ERROR_BASE,				"PKI RSA module on the LLF layer base error" },
}; /* End of PKI_Errors */

/**
****************************************************************
* Data Name: 
* COMMON_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the COMMON module.
***************************************************************/
struct ErrorItem COMMON_Errors[] =
{
/* COMMMON module on the CE2 layer base address - 0x00F00D00 */
{ CE2_COMMON_MODULE_ERROR_BASE,					"COMMON module on the CE2 layer base error" },

/* COMMMON module on the LLF layer base address - 0x00F10D00 */
{ CE2_LLF_COMMON_MODULE_ERROR_BASE,				"COMMON module on the LLF layer base error" },
}; /* End of COMMON_Errors */

/**
****************************************************************
* Data Name: 
* CCM_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the CCM module.
***************************************************************/
struct ErrorItem CCM_Errors[] =
{
/* CCM module on the CE2 layer base address - 0x00F00E00 */
{ CE2_CCM_MODULE_ERROR_BASE,					"CCM module on the CE2 layer base error" },
}; /* End of CCM_Errors */

/**
****************************************************************
* Data Name: 
* INIT_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the INIT module.
***************************************************************/
struct ErrorItem INIT_Errors[] =
{
/* INIT module on the CE2 layer base address - 0x00F00F00 */
{ CE2_INIT_MODULE_ERROR_BASE,					"INIT module on the CE2 layer base error" }
}; /* End of INIT_Errors */

/**
****************************************************************
* Data Name: 
* AESCCM_Errors
*
* \brief \b 
* Description:
*  This structure contains errors of the AESCCM module.
***************************************************************/
struct ErrorItem AESCCM_Errors[] =
{
/********************/
/* AESCCM module error */
/********************/
/* AES module on the CE2 layer base address - 0x00F01500 */
{ CE2_AESCCM_MODULE_ERROR_BASE,					"AESCCM module on the CE2 layer base error" },

/* CE2_AES_CCM() error codes */
{ CE2_AESCCM_ILLEGAL_KEY_SIZE_ERROR,			CE2_KEY_SIZE_ERROR_MESSAGE
												"Supported key size is 128 bits only." },
{ CE2_AESCCM_INVALID_KEY_POINTER_ERROR,			CE2_KEY_PTR_ERROR_MESSAGE },
{ CE2_AESCCM_INVALID_ENCRYPT_MODE_ERROR,		CE2_ENCRYPT_DECRYPT_FLAG_ERROR_MESSAGE },
{ CE2_AESCCM_DATA_IN_POINTER_INVALID_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE },
{ CE2_AESCCM_DATA_OUT_POINTER_INVALID_ERROR,	CE2_DATA_OUT_PTR_ERROR_MESSAGE },
{ CE2_AESCCM_DATA_IN_SIZE_ILLEGAL,				CE2_DATA_IN_SIZE_ERROR_MESSAGE },
{ CE2_AESCCM_DATA_OUT_DATA_IN_OVERLAP_ERROR,	"Input and output pointers overlapped error." },
{ CE2_AESCCM_DATA_OUT_SIZE_INVALID_ERROR,		CE2_DATA_OUT_SIZE_ERROR_MESSAGE
												"Output data buffer not less than "
												"[Input buffer size]+[Tag size] bytes expected." },
{ CE2_AESCCM_ADDITIONAL_BLOCK_NOT_PERMITTED_ERROR, "Additional block not permitted error." },
{ CE2_AESCCM_ILLEGAL_PARAMETER_SIZE_ERROR,		CE2_DATA_IN_SIZE_ERROR_MESSAGE
												"Invalid input parameter size." },
{ CE2_AESCCM_ILLEGAL_PARAMETER_PTR_ERROR,		CE2_DATA_IN_PTR_ERROR_MESSAGE
												"Invalid input parameter pointer."},
{ CE2_AESCCM_ILLEGAL_DATA_TYPE_ERROR,			"Data type is invalid." },
{ CE2_AESCCM_CCM_MAC_INVALID_ERROR,				"Invalid MAC." },
{ CE2_AESCCM_LAST_BLOCK_NOT_PERMITTED_ERROR,	"Last block not permitted error." }
}; /* End of AESCCM_Errors */

/**
****************************************************************
* Data Name: 
* ErrorModules
*
* \brief \b 
* Description:
*  This structure contains errors of the all defined modules.
***************************************************************/
struct ErrorModule ErrorModules[] =
{
	// CE2_AES_ERROR_IDX         0x0UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(AES_Errors),
	// CE2_DES_ERROR_IDX         0x1UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(DES_Errors),
	// CE2_HASH_ERROR_IDX        0x2UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(HASH_Errors),
	// CE2_HMAC_ERROR_IDX        0x3UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(HMAC_Errors),
	// CE2_RSA_ERROR_IDX         0x4UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(RSA_Errors),
	// CE2_DH_ERROR_IDX          0x5UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(DH_Errors),
	// CE2_RC4_ERROR_IDX         0x6UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(RC4_Errors),
	// CE2_INTERNAL_ERROR_IDX    0x7UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(INTERNAL_Errors),
	// CE2_ECPKI_ERROR_IDX       0x8UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(ECPKI_Errors),
	// CE2_CMLA_ERROR_IDX        0x9UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(CMLA_Errors),
	// CE2_CE2_SST_ERROR_IDX     0xAUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(SST_Errors),
	// CE2_PKI_ERROR_IDX         0xBUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(PKI_Errors),
	// CE2_RND_ERROR_IDX         0xCUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(RND_Errors),
	// CE2_COMMON_ERROR_IDX      0xDUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(COMMON_Errors),
	// CE2_CCM_ERROR_IDX         0xEUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(CCM_Errors),
	// CE2_INIT_ERROR_IDX        0xFUL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(INIT_Errors),
	// CE2_C2_ERROR_IDX          0x10UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(C2_Errors),
	// CE2_OTF_ERROR_IDX         0x11UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(OTF_Errors),
	// CE2_KDF_ERROR_IDX         0x12UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(KDF_Errors),
	// CE2_UTILS_ERROR_IDX       0x13UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(UTILS_Errors),
	// CE2_BASE64_ERROR_IDX      0x14UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(Base64_Errors),
	// CE2_AESCCM_ERROR_IDX      0x15UL
	ERRORMODULE_FROM_ERRORITEM_ARRAY(AESCCM_Errors),
	//CE2_LAST_ERROR_IDX        CE2_AESCCM_ERROR_IDX
}; /* End of ErrorModules */

const DxUint32_t ERRORMODULES_COUNT = sizeof(ErrorModules) / sizeof(struct ErrorModule);

/************************ Private function prototype **********/
/************************ Private Functions *******************/

/**
****************************************************************
* Function Name: 
* LLF_GetErrorDescriptionFromErrorModule
*
*  @param aErrorModule [in] - error module in which will be searched
*                             error to retrieve description;
*  @param error        [in] - error code that is used to retrieve
*                             error description.
*
* @returns \b
*  DxUint8_t* - pointer to the specified error description string.
*
* \brief \b 
* Description:
*  This function returns error description from error module
* using error code.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Search error code in input module parameter.
***************************************************************/
DxUint8_t* LLF_GetErrorDescriptionFromErrorModule(
	struct ErrorModule* aErrorModule,	/* in */
	CE2Error_t error)					/* in */
{
	if (aErrorModule != 0 &&
		aErrorModule->errors != 0 && aErrorModule->errors_count != 0)
	{
		DxUint32_t i;

		for (i = 0; i < aErrorModule->errors_count; ++i)
		{
			if (aErrorModule->errors[i].error == error)
			{
				return aErrorModule->errors[i].description;
			}
		}
	}

	return CE2_ERROR_UNDEFINED_ERROR_MESSAGE;
} /* End of LLF_GetErrorDescriptionFromErrorModule */

/************************ Public Functions ********************/

/**
****************************************************************
* Function Name: 
* LLF_GetErrorDescription
*
*  @param error [in] - error code that is used to retrieve error
*                      description.
*
* @returns \b
*  DxUint8_t* - pointer to the specified error description string.
*
* \brief \b 
* Description:
*  This function returns error description using error code.
*
*  \b 
* Algorithm:
*  -# Verify input parameters for validity;
*  -# Call low level function to get error string by error code
*     according input error variable.
***************************************************************/
DxUint8_t* LLF_GetErrorDescription(CE2Error_t error/* in */)
{
	DxUint32_t nErrorLayerIDX;
	DxUint32_t nErrorModuleIDX;

	/* Check for default error code */
	if (error == CE2_OK)
	{
		/* The OK (success) error code */
		return CE2_OK_MESSAGE;
	}

	/* Validate error code */
	nErrorLayerIDX = (error - CE2_ERROR_BASE) / (CE2_ERROR_LAYER_RANGE);

	if (0 > nErrorLayerIDX || nErrorLayerIDX > 1)
		return CE2_ERROR_UNDEFINED_ERROR_LAYER_MESSAGE;

	nErrorModuleIDX =
		(error - CE2_ERROR_BASE - nErrorLayerIDX*CE2_ERROR_LAYER_RANGE) /
		(CE2_ERROR_MODULE_RANGE);

	if (0 > nErrorModuleIDX || nErrorModuleIDX > CE2_LAST_ERROR_MODULE_IDX)
		return CE2_ERROR_UNDEFINED_ERROR_MODULE_MESSAGE;

	if (nErrorModuleIDX >= ERRORMODULES_COUNT)
		return CE2_ERROR_UNDEFINED_ERROR_MESSAGE;

	return LLF_GetErrorDescriptionFromErrorModule(&ErrorModules[nErrorModuleIDX], error);
} /* End of LLF_GetErrorDescription */
